# -*- coding: utf-8 -*-
"""
Sizing batteries for power flow management in distribution grids

Created on Wed Oct  4 11:11:56 2023

@author: Chrisian van Someren

Notes:
    1. At the transformer, positive values are net production, negative
        values are net consumption. At nodes, this is opposite.
    2. For the battery, a negative power is discharge, a positive value is charge.

"""

#load the relevent modules
import pandapower as pp
import numpy as np
import pandas as pd
import pandapower.control as control
import pandapower.timeseries as ts
from pandapower.timeseries.data_sources.frame_data import DFData
from pandapower.timeseries import OutputWriter
from pandapower.timeseries.run_time_series import run_timeseries
from pandapower.plotting.plotly.mapbox_plot import set_mapbox_token
import math
import matplotlib.pyplot as plt
import os
import sys
import warnings
import time
import winsound
import networkx as nx
import pandapower.topology as top
import random as random
warnings.simplefilter(action='ignore', category=FutureWarning)
set_mapbox_token('<token>')
pd.set_option('display.max_rows', 1000)
pd.set_option('display.max_columns', 500)
pd.set_option('display.width', 200)
np.set_printoptions(threshold=sys.maxsize)

def Batch_2():
    Capacities = np.empty(shape=(5, 8), dtype='float')
    load_pattern = 'IEEE'
    storage = '0' #'CES'
    for n_DES in range(1,6):
        for load_pattern in range(3,9):
            if load_pattern == 3:
                load_pattern = r"[directory]\Battery Optimization Model\Load Pattern_30.csv"
            elif load_pattern == 4:
                load_pattern = r"[directory]\Load Pattern_40.csv"
            elif load_pattern == 5:
                load_pattern = r"[directory]\Load Pattern_50.csv"
            elif load_pattern == 6:
                load_pattern = r"[directory]\Load Pattern_60.csv"
            elif load_pattern == 7:
                load_pattern = r"[directory]\Load Pattern_70.csv"
            elif load_pattern == 8:
                load_pattern = r"[directory]\Load Pattern_80.csv"
            for i in range(2,3):
                if i == 0: 
                    impedence_mul = 0.01
                elif i == 1:
                    impedence_mul = 0.5
                elif i == 2:
                    impedence_mul = 1.0
                elif i == 3:
                    impedence_mul = 1.5
                else:
                    impedence_mul = 2.0
                print('layout %s, simultaneity %s, length multiplier x%s' % (storage, load_pattern/10, impedence_mul))
                peak_capacity = Run(load_pattern, storage, impedence_mul, simultaneity=load_pattern)
                Capacities[i,(load_pattern-3)] = peak_capacity
                print("%s capacities:" % storage)
                print(Capacities)

def Run(load_pattern = 'IEEE', storage = '0',impedence_mul = 1,n_DES=0, simultaneity=0):
    input_file = r"[directory]\IEEE Test Grid.xlsx" 
    output_dir = r"[directory]"
    gen_pattern = r"[directory]"
    Initialize = True           #Set to True to set up grid model
    
    if storage == 'CES':
        Implement_CES = True       #Set to True to model a CES battery system - Mutually exclusive with DES!
        Implement_DES = False       #Set to True to model a DES battery system - Mutually exclusive with CES!        
    elif storage=='DES':
        Implement_CES = False       #Set to True to model a CES battery system - Mutually exclusive with DES!
        Implement_DES = True       #Set to True to model a DES battery system - Mutually exclusive with CES!        
    else:
        Implement_CES = False       #Set to True to model a CES battery system - Mutually exclusive with DES!
        Implement_DES = False       #Set to True to model a DES battery system - Mutually exclusive with CES!    

    Roundtrip_Efficiency = 1  #Define a (constant) efficiency value for battery (dis)charging
    #Safety_margin = 0.95       #Define the maximum desired loading of components
    Time_step = 0.25            #Define number of hours per time step
    allowed_error = 0.001        #Error allowed when minimizing battery capacity - FINAL RESULTS SHOULD USE 0.0001 but takes too long, esp. DES
    Capacity_limit = 10     #The network load limit, used to calculate the safety factor, in kW/house
    cable_length_multiplyer = 1
    total_capacity = 300 #kWh

    if Initialize == True:
        net, Safety_margin, DES_loc = Generate_ieee_grid(input_file, impedence_mul)
        # Save and plot grid model
        pp.to_pickle(net, "WorkingDoc.p")
        #pp.diagnostic(net)
        # Initial State
        net = Run_Model(net, load_pattern, gen_pattern, output_dir)
        #Plot_Grid(net)
    else:
        # Load grid model
        net = pp.from_pickle("WorkingDoc.p", convert=True)
        
    if Implement_CES == True:
        net = pp.from_pickle("WorkingDoc.p", convert=True)
        # Create a Battery
        net, battery_pattern, peak_capacity = Create_CES_ieee(net, output_dir, Roundtrip_Efficiency, 
                                                                  Safety_margin, Time_step, allowed_error, load_pattern, gen_pattern)
        
        battery_pattern.to_excel(output_dir+'/battery_pattern_'+str(simultaneity)+'.xlsx',sheet_name=(storage+'_'+str(impedence_mul)+'_'+str(simultaneity)))
        #Plot_Grid(net)
        net = Run_Model(net, load_pattern, gen_pattern, output_dir, battery_pattern, True)       
        #Plot_Grid(net)
    
    elif Implement_DES == True:
        net = pp.from_pickle("WorkingDoc.p", convert=True)
        net, battery_pattern, peak_capacity = Create_DES_ieee(net, output_dir, Roundtrip_Efficiency, 
                                                                  Safety_margin, Time_step, allowed_error, load_pattern, 
                                                                  gen_pattern, n_DES, DES_loc)
        net = Run_Model(net, load_pattern, gen_pattern, output_dir, battery_pattern, True)
    
    Run_Complete()
    return peak_capacity

def Generate_ieee_grid(input_file, length_mul=1):
    #Read in data
    df = pd.read_excel(input_file, sheet_name = 'Lines')
    cable_data = df.to_numpy()
    df = pd.read_excel(input_file, sheet_name = 'Transformer')
    trafo_data = df.to_numpy()
    df = pd.read_excel(input_file, sheet_name = 'Buscoords')
    Coords = df.to_numpy()
    df = pd.read_excel(input_file, sheet_name = 'LineCodes')
    LineCodes = df.to_numpy()
    df = pd.read_excel(input_file, sheet_name = 'Loads')
    loads = df.to_numpy()
    #print(loads)
    net = pp.create_empty_network(name="IEEE Test Grid", f_hz=50.0)
    
    #Create lines
    for linetype in range(1,len(LineCodes)):
        pp.create_std_type(net, {"c_nf_per_km": float(LineCodes[linetype,6]), 
                                 "r_ohm_per_km": (float(LineCodes[linetype,2])*length_mul),
                                 "x_ohm_per_km": (float(LineCodes[linetype,3])*length_mul),
                                 "c0_nf_per_km": float(LineCodes[linetype,7]),
                                 "r0_ohm_per_km": float(LineCodes[linetype,4]),
                                 "x0_ohm_per_km": float(LineCodes[linetype,5]),
                                 "max_i_ka": (400/np.sqrt((float(LineCodes[linetype,2])**2 + float(LineCodes[linetype,3])**2))/1000)}, 
                           LineCodes[linetype,0], element='line')
                            #(400/float(LineCodes[linetype,3])/1000)},
    
    #Create Trafo
    pp.create_std_type(net, {"sn_mva": 0.8,
        "vn_hv_kv": 11,
        "vn_lv_kv": 0.416,
        "vk_percent": 4,
        "vkr_percent": 0.4,
        "pfe_kw": 0,
        "i0_percent": 0,
        "shift_degree": 30, #150
        #"vector_group": vector_group,
        "tap_side": "hv",
        "tap_neutral": 0,
        "tap_min": -2,
        "tap_max": 2,
        "tap_step_degree": 0,
        "tap_step_percent": 2.5,
        "tap_phase_shifter": False,
        },name='IEEE_trafo', element="trafo")
    
    # Low resistance line
    line_data = {"c_nf_per_km": 1, "r_ohm_per_km": 0.001, "x_ohm_per_km": 0.0001, "max_i_ka":10}
    pp.create_std_type(net, line_data, 'Low Resistance', element='line')
    
    #Create Trafo
    #Add coords...
    x_coord = float(Coords[1,1])
    y_coord = float(Coords[1,2])
    bus_coords = (x_coord, y_coord)
    mv_bus = pp.create_bus(net, name = "Bus_MV", vn_kv=11, type='n',geodata=bus_coords)
    lv_bus = pp.create_bus(net, name = "Bus_LV_trafo", vn_kv=0.416, type='n',geodata=bus_coords)
    pp.create_ext_grid(net, mv_bus, vm_pu=1.05, va_degrees=0, name="External_grid")
    pp.create_transformer(net, mv_bus, lv_bus, std_type = 'IEEE_trafo', name = "MV-LV-Trafo")
    pp.create_switch(net, mv_bus, mv_bus, et='t', closed=True, name='Switch MV')

    #Create Buses
    DES_loc = []
    row=0
    load_bus_nr=0
    for bus in range(2,907):
        x_coord = float(Coords[bus,1])
        y_coord = float(Coords[bus,2])
        bus_coords = (x_coord, y_coord)
        bus_nr = pp.create_bus(net, name='Bus_LV%s' % bus, vn_kv=0.416, type='n',
                      geodata=bus_coords)
        if (bus) == int(loads[load_bus_nr+2,2]):
            pp.create_load(net=net, bus=bus, p_mw=0.000001, q_mvar=0.000001, 
                           name = 'Load_LV%s' % load_bus_nr)
            pp.create_sgen(net=net, bus=bus, p_mw=0.000001, q_mvar=0.000001, 
                           name = 'Gen_LV%s' % load_bus_nr)
            DES_loc.append([bus_nr,0,0,0,0])
            #row+=1
            if load_bus_nr < 54:    
                load_bus_nr+=1

    #Create Lines
    for line in range(1,906):
        from_bus = int(cable_data[line,1])
        to_bus = int(cable_data[line,2])
        length = float(cable_data[line,4])/1000 #* length_mul
        line_name = ('LV_line%s' % cable_data[line,0])
        line_type = cable_data[line,6]
        pp.create_line(net, from_bus, to_bus, length_km=length,
                   std_type=line_type, name=line_name)
    
    #Generate Switches  
    lv_buses = net.bus[net.bus.vn_kv == 0.416]
    lv_ls = net.line[(net.line.from_bus.isin(lv_buses.index)) & (net.line.to_bus.isin(lv_buses.index))]
    for _, line in lv_ls.iterrows():
            pp.create_switch(net, line.from_bus, line.name, et='l', closed=True, type='LBS', 
                             name='Switch_%s.%s' % (net.bus.name.at[line.from_bus], line['name']))
            pp.create_switch(net, line.to_bus, line.name, et='l', closed=True, type='LBS', 
                             name='Switch_%s.%s' % (net.bus.name.at[line.to_bus], line['name']))
    
    # Determine the max loading of the transformer, standardized per house
    Safety_margin=np.empty(shape=(1, 1), dtype='object')
    P_trafo_max = net.trafo.sn_mva.to_numpy()
    Safety_margin[0,0] = 0.95#Capacity_limit/Grid_capacity
    DES_loc = np.array(DES_loc,dtype='float')
    
    return net, Safety_margin, DES_loc

def Run_Model(net, load_pattern, gen_pattern, output_dir, battery_pattern=0, run_storage=False):  
    #time.sleep(1)
    
    #Load time series from csv
    df_load = pd.read_csv(load_pattern)
    df_load = pd.DataFrame(df_load)
    ds_load = ts.DFData(df_load, multi=True)
    
    df_gen = pd.read_csv(gen_pattern)
    df_gen = pd.DataFrame(df_gen)
    ds_gen = ts.DFData(df_gen, multi=True)
    
    if run_storage == True:
        #Convert array to dataframe
        df_battery = pd.DataFrame(battery_pattern)
        #Convert  dataframe to be used by pandapower
        ds_battery = ts.DFData(df_battery, multi=True)
    
    #Number of time steps
    n_ts = range(0,len(df_load))
    
    #print("n_ts: %s" % n_ts)
        
    #Build an Outwriter to view results
    ow = OutputWriter(net, n_ts, output_path=output_dir, output_file_type=".xlsx", log_variables=list()) #csv_separator=","
    # adding vm_pu of all buses and line_loading in percent of all lines as outputs to be stored
    #Can also log for example: net.res_line.loc[net.res_line.loading_percent > 50]
    ow.log_variable('res_bus', 'vm_pu')
    ow.log_variable('res_line', 'loading_percent')
    ow.log_variable('res_line', 'i_ka')
    ow.log_variable('res_line', 'p_to_mw')
    ow.log_variable('res_load', 'p_mw')
    ow.log_variable('res_bus', 'p_mw')#, index=lv_buses_index, eval_function=np.sum, eval_name="lv_bus_sum_p")
    ow.log_variable('res_bus', 'vm_pu')#, index=lv_buses_index, eval_function=np.max, eval_name="lv_bus_max")
    ow.log_variable('res_trafo', 'loading_percent')
    ow.log_variable('res_trafo', 'p_lv_mw')
    ow.log_variable("res_sgen", "p_mw")
    ow.log_variable('res_trafo','p_hv_mw')
    ow.log_variable('res_trafo','pl_mw')
    ow.log_variable('res_trafo','ql_mvar')
    ow.log_variable('res_trafo','q_hv_mvar')
    ow.log_variable('res_trafo','q_lv_mvar')

    #Run battery
    if run_storage == True:
        #Create load pattern - feed in battery control dataframe
        const_storage = control.ConstControl(net, element='storage', element_index=net.storage.index, 
                                                 variable='p_mw', data_source=ds_battery, profile_name=net.storage.index)
        ow.log_variable("storage", "p_mw")
        ow.log_variable("storage", "soc_percent")

    const_load = control.ConstControl(net, element='load', element_index=net.load.index, 
                                          variable='p_mw', data_source=ds_load, profile_name=net.load.index)
    const_gen = control.ConstControl(net, element='sgen', element_index=net.load.index, 
                                          variable='p_mw', data_source=ds_gen, profile_name=net.load.index)
    
    run_timeseries(net, time_steps=n_ts) #, continue_on_divergence=True)
    return net

def Create_CES_ieee(net, output_dir, Roundtrip_Efficiency, Safety_margin, Time_step, allowed_error, load_pattern, gen_pattern):
    line_file = os.path.join(output_dir, "res_line", "loading_percent.xlsx")
    line_file_2 = os.path.join(output_dir, "res_line", "p_to_mw.xlsx")
    P_trafo_max = net.trafo.sn_mva.to_numpy() # MW 
    df = pd.read_excel(line_file)
    line_load_percent = df.to_numpy()
    print(line_load_percent[3,1])
    df = pd.read_excel(line_file_2)
    line_load_mw = df.to_numpy()
    print(line_load_mw[3,1])
    n_batteries = 1
    
    print('n_batteries: %s' % n_batteries)
    time_steps = len(df)   
    battery_pattern = np.empty(shape=(time_steps, n_batteries), dtype='object')
    
    for battery in range(0,n_batteries):
        P_trafo_max[battery] *= Safety_margin[battery,0]
        P_line_max_percent = 45
        P_trafo_max[battery] = line_load_mw[3,1]/line_load_percent[3,1] * P_line_max_percent
        P_line_max_mw = line_load_mw[3,1]/line_load_percent[3,1] * P_line_max_percent
        P_battery_max = 0
        #Determine minimum battery power requirements
        for i in range(0, time_steps):
            if abs(line_load_percent[i,1]) <= P_line_max_percent: #Do nothing
                    battery_pattern[i,battery] = 0
            else:
                if line_load_percent[i,1] > 0: #Surplus production
                    battery_pattern[i,battery] = line_load_mw[i,1] - P_line_max_mw
                else: #Too much consumption
                    battery_pattern[i,battery] = -1 * (line_load_mw[i,1] - P_line_max_mw)
            if abs(battery_pattern[i,battery]) > P_battery_max:
                P_battery_max = abs(battery_pattern[i,battery])
        battery_capacity=0
        peak_capacity=0
        battery_max_capacity=0
        storage_capacity = 0
        storage_discharge = 0
        #Determine Battery Capacity
        #A) In each time slot, determine what the battery charge needs to be
        for i in range(0,time_steps):
            #print("battery max capacity: %s %s" % (battery_max_capacity, type(battery_max_capacity)))
            if battery_pattern[i,battery] > 0: #Charge
                battery_max_capacity += battery_pattern[i,battery] * Time_step
            elif battery_pattern[i,battery] < 0: #Discharge
                battery_max_capacity += battery_pattern[i,battery] * Time_step
            elif battery_max_capacity > 0: #Reset by discharging 
                battery_max_capacity -= abs(line_load_mw[i,1]) * Time_step
                if battery_max_capacity < 0: battery_max_capacity = 0
            elif battery_max_capacity < 0: #Reset by charging
                battery_max_capacity += (P_trafo_max[battery] + line_load_mw[i,1]) * Time_step
                if battery_max_capacity > 0: battery_max_capacity = 0
            if abs(battery_max_capacity) > peak_capacity:
                peak_capacity = abs(battery_max_capacity)
        
        WriteFile = os.path.join(output_dir, "Analysis", "Battery Charge Pattern Check CES.csv")
        df_battery = pd.DataFrame(battery_pattern)
        df_battery.to_csv(WriteFile)
        
        battery_bus = pp.get_element_index(net, "bus",'Bus_LV%s' % 2) #2 / 114 / 899 / 861 / 101 / 280 / 472 / 654 / 840
        pp.create_storage(net, bus=battery_bus, p_mw=P_battery_max, max_e_mwh=peak_capacity)
        battery_pattern_df = pd.DataFrame(battery_pattern)
        
    # Minimize Solution -------------------------------------------------------------------------------------
    net = Run_Model(net, load_pattern, gen_pattern, output_dir, battery_pattern, True) 
    peak_capacity = peak_capacity/Roundtrip_Efficiency
    print("Peak Capacity for battery #%s: %s kWh (incl. roundtrip efficiency)" % (battery, peak_capacity*1000))    

    
    line_file = os.path.join(output_dir, "res_line", "loading_percent.xlsx")
    line_file_2 = os.path.join(output_dir, "res_line", "p_to_mw.xlsx")
    P_trafo_max = net.trafo.sn_mva.to_numpy() # MW 
    df = pd.read_excel(line_file)
    line_load_percent = df.to_numpy()
    print('load percent: %s' % line_load_percent[3,1])
    df = pd.read_excel(line_file_2)
    line_load_mw = df.to_numpy()
    print('load mw: %s' % line_load_mw[3,1])
        
    while line_load_percent[3,1] > P_line_max_percent * (1+allowed_error) or \
        line_load_percent[3,1] < P_line_max_percent * (1-allowed_error):
        error = (line_load_percent[3,1]-P_line_max_percent) / P_line_max_percent
        print('error: %s' % error)
        for i in range(0, time_steps):
            battery_pattern[i,0] *= error
        peak_capacity = battery_pattern.sum()*Time_step*-1
        df_battery = pd.DataFrame(battery_pattern)
        df_battery.to_csv(WriteFile)
        battery_pattern_df = pd.DataFrame(battery_pattern)
        net = Run_Model(net, load_pattern, gen_pattern, output_dir, battery_pattern, True) 
        peak_capacity = peak_capacity/Roundtrip_Efficiency
        print("Peak Capacity for battery #%s: %s kWh (incl. roundtrip efficiency)" % (battery, peak_capacity*1000))    
        
        line_file = os.path.join(output_dir, "res_line", "loading_percent.xlsx")
        line_file_2 = os.path.join(output_dir, "res_line", "p_to_mw.xlsx")
        P_trafo_max = net.trafo.sn_mva.to_numpy() # MW 
        df = pd.read_excel(line_file)
        line_load_percent = df.to_numpy()
        print(line_load_percent[3,1])
        df = pd.read_excel(line_file_2)
        line_load_mw = df.to_numpy()
        print(line_load_mw[3,1])

    return net, battery_pattern_df, peak_capacity*1000

def Create_DES_ieee(net, output_dir, Roundtrip_Efficiency, Safety_margin, Time_step, allowed_error, load_pattern, gen_pattern, n_DES, DES_loc):
    line_file = os.path.join(output_dir, "res_line", "loading_percent.xlsx")
    line_file_2 = os.path.join(output_dir, "res_line", "p_to_mw.xlsx")
    P_trafo_max = net.trafo.sn_mva.to_numpy() # MW 
    df = pd.read_excel(line_file)
    line_load_percent = df.to_numpy()
    print(line_load_percent[3,1])
    df = pd.read_excel(line_file_2)
    line_load_mw = df.to_numpy()
    print(line_load_mw[3,1])
    Peak_line_load_mw=line_load_mw[3,1]
    n_batteries = n_DES
    
    print('n_batteries: %s' % n_batteries)
    time_steps = len(df)   
    battery_pattern = np.empty(shape=(time_steps, n_batteries), dtype='object')
    P_battery_max = np.empty(shape=(n_batteries), dtype='object')
    
    #Test battery "effectiveness"
    Cummulative_effect=0
    for battery in range(0,n_batteries):
        Effect=0
        #Test each battery's effect
        print('test completion: %s' % (battery/n_batteries))
        battery_pattern.fill(0)
        battery_pattern[2,battery] = -0.01
        battery_pattern_df = pd.DataFrame(battery_pattern)
        net = Run_Model(net, load_pattern, gen_pattern, output_dir, battery_pattern_df, True)
        df = pd.read_excel(line_file_2)
        line_load_mw = df.to_numpy()
        Effect = line_load_mw[3,1] - Peak_line_load_mw
        Cummulative_effect += Effect
        DES_loc[battery,1] = Effect
        
    print('Test Results')
    print(DES_loc)

    #Define battery "effectiveness" as 'scalar' variable
    for battery in range(0,n_batteries):
        print('battery: %s' % battery)
        print('safety: %s' % Safety_margin[0,0])
        scalar = 0
        if battery == 0:
            scalar = 0.5
        elif battery == 1:
            scalar == 0.45
        elif battery == 2:
            scalar = 0.05
        elif battery == 3:
            scalar = 0.04
        elif battery == 4:
            scalar = 0.02
        
        P_trafo_max[0] *= Safety_margin[0,0]
        P_line_max_percent = 45
        P_trafo_max[0] = line_load_mw[3,1]/line_load_percent[3,1] * P_line_max_percent
        P_line_max_mw = line_load_mw[3,1]/line_load_percent[3,1] * P_line_max_percent
        P_battery_max[battery] = 0
        #Determine minimum battery power requirements
        for i in range(0, time_steps):
            if abs(line_load_percent[i,1]) <= P_line_max_percent: #Do nothing
                    battery_pattern[i,battery] = 0
            else:
                if line_load_percent[i,1] > 0: #Surplus production
                    battery_pattern[i,battery] = (line_load_mw[i,1] - P_line_max_mw) * scalar
                else: #Too much consumption
                    battery_pattern[i,battery] = -1 * (line_load_mw[i,1] - P_line_max_mw) * scalar
            if abs(battery_pattern[i,battery]) > P_battery_max[battery]:
                P_battery_max[battery] = abs(battery_pattern[i,battery])
        battery_capacity=0
        peak_capacity=0
        battery_max_capacity=0
        storage_capacity = 0
        storage_discharge = 0
        #Determine Battery Capacity
        #A) In each time slot, determine what the battery charge needs to be
        for i in range(0,time_steps):
            #print("battery max capacity: %s %s" % (battery_max_capacity, type(battery_max_capacity)))
            if battery_pattern[i,battery] > 0: #Charge
                battery_max_capacity += battery_pattern[i,battery] * Time_step
            elif battery_pattern[i,battery] < 0: #Discharge
                battery_max_capacity += battery_pattern[i,battery] * Time_step
            elif battery_max_capacity > 0: #Reset by discharging 
                battery_max_capacity -= abs(line_load_mw[i,1]) * Time_step
                if battery_max_capacity < 0: battery_max_capacity = 0
            elif battery_max_capacity < 0: #Reset by charging
                battery_max_capacity += (P_trafo_max[0] + line_load_mw[i,1]) * Time_step * scalar
                if battery_max_capacity > 0: battery_max_capacity = 0
            if abs(battery_max_capacity) > peak_capacity:
                peak_capacity = abs(battery_max_capacity)

        WriteFile = os.path.join(output_dir, "Analysis", "Battery Charge Pattern Check CES.csv")
        df_battery = pd.DataFrame(battery_pattern)
        df_battery.to_csv(WriteFile)
        
        #Define battery locations
        if battery == 0:
            battery_bus = pp.get_element_index(net, "bus",'Bus_LV%s' % 899) #2 / 114 / 899 / 861 / 101 / 280 / 472 / 654 / 840
        elif battery == 1:
            battery_bus = pp.get_element_index(net, "bus",'Bus_LV%s' % 639)
        elif battery == 2:
            battery_bus = pp.get_element_index(net, "bus",'Bus_LV%s' % 320)
        elif battery == 3:
            battery_bus = pp.get_element_index(net, "bus",'Bus_LV%s' % 249)
        elif battery == 4:
            battery_bus = pp.get_element_index(net, "bus",'Bus_LV%s' % 74)
            
        pp.create_storage(net, bus=battery_bus, p_mw=P_battery_max[battery], max_e_mwh=peak_capacity)
        battery_pattern_df = pd.DataFrame(battery_pattern)
        
        # Minimize Solution -------------------------------------------------------------------------------------
    net = Run_Model(net, load_pattern, gen_pattern, output_dir, battery_pattern, True) 
    peak_capacity = P_battery_max.sum()/Roundtrip_Efficiency
    print("Peak Capacity for battery #%s: %s kWh (incl. roundtrip efficiency)" % (battery, peak_capacity*1000))    

    line_file = os.path.join(output_dir, "res_line", "loading_percent.xlsx")
    line_file_2 = os.path.join(output_dir, "res_line", "p_to_mw.xlsx")
    P_trafo_max[0] = net.trafo.sn_mva.to_numpy() # MW 
    df = pd.read_excel(line_file)
    line_load_percent = df.to_numpy()
    print('load percent: %s' % line_load_percent[3,1])
    df = pd.read_excel(line_file_2)
    line_load_mw = df.to_numpy()
    print('load mw: %s' % line_load_mw[3,1])
        
    while line_load_percent[3,1] > P_line_max_percent * (1+allowed_error) or \
        line_load_percent[3,1] < P_line_max_percent * (1-allowed_error):
        error = (line_load_percent[3,1] - P_line_max_percent) / P_line_max_percent
        print('error: %s' % error)
        for battery in range(0,n_batteries):
            for i in range(0, time_steps):
                battery_pattern[i,battery] *= error
            peak_capacity = battery_pattern.sum()*Time_step*-1
        df_battery = pd.DataFrame(battery_pattern)
        df_battery.to_csv(WriteFile)
        battery_pattern_df = pd.DataFrame(battery_pattern)
        net = Run_Model(net, load_pattern, gen_pattern, output_dir, battery_pattern, True) 
        peak_capacity = peak_capacity/Roundtrip_Efficiency
        print("Peak Capacity for battery #%s: %s kWh (incl. roundtrip efficiency)" % (battery, peak_capacity*1000))    
        
        line_file = os.path.join(output_dir, "res_line", "loading_percent.xlsx")
        line_file_2 = os.path.join(output_dir, "res_line", "p_to_mw.xlsx")
        P_trafo_max[0] = net.trafo.sn_mva.to_numpy() # MW 
        df = pd.read_excel(line_file)
        line_load_percent = df.to_numpy()
        print(line_load_percent[3,1])
        df = pd.read_excel(line_file_2)
        line_load_mw = df.to_numpy()
        print(line_load_mw[3,1])

    return net, battery_pattern_df, peak_capacity*1000

def Plot_Grid(net):
    #Plot grid
    print('Plotting...')
    from pandapower.plotting import simple_plot, simple_plotly, pf_res_plotly, pf_res_plotly
    pf_res_plotly(net, figsize = 5)
    
def Run_Complete():
    print("Done!")